\
// ---------------------------------------------------------
// Nano Ardule - Step 04b: Program Change via Encoder + Note Loop (v2 FIX)
// ---------------------------------------------------------

#include <Wire.h>
#include <LiquidCrystal_I2C.h>

#define MIDI_BAUD 31250
#define ACT_LED   9

#define LCD_ADDR 0x27
LiquidCrystal_I2C lcd(LCD_ADDR, 16, 2);

const uint8_t ENC_CLK = 2;
const uint8_t ENC_DT  = 3;
const uint8_t ENC_SW  = 4;

#ifndef STEP_GUARD_US
#define STEP_GUARD_US 800
#endif

volatile long encSteps = 0;
volatile unsigned long lastStepUs = 0;

void onEncCLKRise() {
  unsigned long now = micros();
  if ((unsigned long)(now - lastStepUs) < STEP_GUARD_US) return;
  lastStepUs = now;
  if (digitalRead(ENC_DT) == LOW) encSteps++; else encSteps--;
}

const unsigned long DEBOUNCE_MS = 25;
const unsigned long LONG_MS     = 700;
bool sw_lastStable = true;
unsigned long sw_pressMs = 0;
bool sw_longReported = false;

uint8_t  currentPrg = 0;
bool     pendingSend = false;
unsigned long lastMoveMs = 0;
const unsigned long COMMIT_IDLE_MS = 200;

bool transientMsg = false;
unsigned long lastMsgMs = 0;

bool loopEnabled = false;
const uint8_t CH_A = 1;
const uint8_t noteSeq[] = {60, 64, 67, 72};
const uint8_t noteCount = sizeof(noteSeq)/sizeof(noteSeq[0]);
uint8_t noteIndex = 0;
bool noteOnActive = false;
unsigned long noteTimestampMs = 0;
const unsigned long NOTE_LEN_MS = 300;
const unsigned long NOTE_GAP_MS = 120;

void sendByte(uint8_t b){ Serial.write(b); }
void sendProgramChange(uint8_t channel, uint8_t program){
  sendByte(0xC0 | ((channel-1) & 0xF));
  sendByte(program & 0x7F);
}
void sendNoteOn(uint8_t channel, uint8_t note, uint8_t vel){
  sendByte(0x90 | ((channel-1) & 0xF));
  sendByte(note & 0x7F);
  sendByte(vel & 0x7F);
}
void sendNoteOff(uint8_t channel, uint8_t note, uint8_t vel){
  sendByte(0x80 | ((channel-1) & 0xF));
  sendByte(note & 0x7F);
  sendByte(vel & 0x7F);
}
void activity(){ digitalWrite(ACT_LED, HIGH); delay(8); digitalWrite(ACT_LED, LOW); }

void lcdLineClear(uint8_t row){ lcd.setCursor(0,row); lcd.print("                "); }
void lcdLinePrint16(uint8_t row, const char* s){
  char buf[17]; snprintf(buf, sizeof(buf), "%-16s", s);
  lcd.setCursor(0,row); lcd.print(buf);
}
void lcdShowProgram(){
  char line0[17];
  snprintf(line0, sizeof(line0), "PRG:%3u          ", currentPrg);
  lcdLineClear(0); lcdLinePrint16(0, line0);
}
void lcdShowLoop(){
  lcdLineClear(1);
  lcdLinePrint16(1, loopEnabled ? "Loop: ON " : "Loop: OFF");
}

void scanButton(){
  unsigned long now = millis();
  bool raw = digitalRead(ENC_SW);
  static bool lastRaw = true;
  static unsigned long lastRawMs = 0;

  if (raw != lastRaw){ lastRaw = raw; lastRawMs = now; }
  if (now - lastRawMs < DEBOUNCE_MS) return;

  if (raw != sw_lastStable){
    sw_lastStable = raw;
    if (!sw_lastStable){
      sw_pressMs = now; sw_longReported = false;
    } else {
      if (!sw_longReported){
        sendProgramChange(CH_A, currentPrg);
        activity();
        lcdLineClear(1); lcdLinePrint16(1, "Sent PC (SHORT)");
        transientMsg = true; lastMsgMs = millis();
      }
    }
  }
  if (!sw_lastStable && !sw_longReported && (now - sw_pressMs) >= LONG_MS){
    sw_longReported = true;
    loopEnabled = !loopEnabled;
    if (!loopEnabled && noteOnActive){
      uint8_t n = noteSeq[(noteIndex + noteCount - 1) % noteCount];
      sendNoteOff(CH_A, n, 64);
      noteOnActive = false;
    }
    lcdShowLoop();
  }
}

void runNoteLoop(){
  if (!loopEnabled) return;
  unsigned long now = millis();
  if (!noteOnActive){
    if (now - noteTimestampMs >= NOTE_GAP_MS){
      uint8_t n = noteSeq[noteIndex];
      noteIndex = (noteIndex + 1) % noteCount;
      sendNoteOn(CH_A, n, 100);
      activity();
      noteOnActive = true;
      noteTimestampMs = now;
    }
  } else {
    if (now - noteTimestampMs >= NOTE_LEN_MS){
      uint8_t n = noteSeq[(noteIndex + noteCount - 1) % noteCount];
      sendNoteOff(CH_A, n, 64);
      noteOnActive = false;
      noteTimestampMs = now;
    }
  }
}

void setup(){
  pinMode(ACT_LED, OUTPUT); digitalWrite(ACT_LED, LOW);

  Wire.begin();
  lcd.init(); lcd.noAutoscroll(); lcd.noCursor(); lcd.noBlink(); lcd.backlight(); lcd.clear();
  lcdLinePrint16(0, "PRG:  0         ");
  lcdLinePrint16(1, "Loop: OFF       ");
  delay(30); lcdLineClear(0); lcdLineClear(1);
  lcdLinePrint16(0, "PRG:  0         ");
  lcdLinePrint16(1, "Loop: OFF       ");

  pinMode(ENC_CLK, INPUT_PULLUP);
  pinMode(ENC_DT,  INPUT_PULLUP);
  pinMode(ENC_SW,  INPUT_PULLUP);
  attachInterrupt(digitalPinToInterrupt(ENC_CLK), onEncCLKRise, RISING);

  Serial.begin(MIDI_BAUD);
  delay(300);
}

void loop(){
  unsigned long now = millis();

  long steps; noInterrupts(); steps = encSteps; encSteps = 0; interrupts();
  if (steps != 0){
    long v = (long)currentPrg + steps;
    if (v < 0) v = 0; else if (v > 127) v = 127;
    if ((uint8_t)v != currentPrg){
      currentPrg = (uint8_t)v;
      lcdShowProgram();
    }
    pendingSend = true;
    lastMoveMs = now;
  }

  if (pendingSend && (now - lastMoveMs) >= COMMIT_IDLE_MS){
    pendingSend = false;
    sendProgramChange(CH_A, currentPrg);
    activity();
    if (!transientMsg) lcdShowLoop();
  }

  scanButton();
  runNoteLoop();

  if (transientMsg && (millis() - lastMsgMs >= 1000)){
    transientMsg = false;
    lcdShowLoop();
  }

  delay(1);
}
