\
// ---------------------------------------------------------
// Nano Ardule - Step 04: Program Change via Encoder
// ---------------------------------------------------------

#include <Wire.h>
#include <LiquidCrystal_I2C.h>

#define MIDI_BAUD 31250
#define ACT_LED   9

#define LCD_ADDR 0x27
LiquidCrystal_I2C lcd(LCD_ADDR, 16, 2);

const uint8_t ENC_CLK = 2;
const uint8_t ENC_DT  = 3;
const uint8_t ENC_SW  = 4;

#ifndef STEP_GUARD_US
#define STEP_GUARD_US 800
#endif

volatile long encSteps = 0;
volatile unsigned long lastStepUs = 0;

void onEncCLKRise() {
  unsigned long now = micros();
  if ((unsigned long)(now - lastStepUs) < STEP_GUARD_US) return;
  lastStepUs = now;
  if (digitalRead(ENC_DT) == LOW) encSteps++; else encSteps--;
}

const unsigned long DEBOUNCE_MS = 25;
const unsigned long LONG_MS     = 700;
bool sw_lastStable = true;
unsigned long sw_pressMs = 0;
bool sw_longReported = false;

uint8_t  currentPrg = 0;
bool     pendingSend = false;
unsigned long lastMoveMs = 0;
const unsigned long COMMIT_IDLE_MS = 200;

void sendByte(uint8_t b){ Serial.write(b); }
void sendProgramChange(uint8_t channel, uint8_t program){
  sendByte(0xC0 | ((channel-1) & 0x0F));
  sendByte(program & 0x7F);
}
void activity(){
  digitalWrite(ACT_LED, HIGH); delay(8); digitalWrite(ACT_LED, LOW);
}

void lcdLineClear(uint8_t row){
  lcd.setCursor(0,row); lcd.print("                ");
}
void lcdLinePrint16(uint8_t row, const char* s){
  char buf[17]; snprintf(buf, sizeof(buf), "%-16s", s);
  lcd.setCursor(0,row); lcd.print(buf);
}
void lcdShowProgram(){
  char line0[17];
  snprintf(line0, sizeof(line0), "PRG:%3u          ", currentPrg);
  lcdLineClear(0);
  lcdLinePrint16(0, line0);
}

void scanButton(){
  unsigned long now = millis();
  bool raw = digitalRead(ENC_SW);
  static bool lastRaw = true;
  static unsigned long lastRawMs = 0;

  if (raw != lastRaw){ lastRaw = raw; lastRawMs = now; }
  if (now - lastRawMs < DEBOUNCE_MS) return;

  if (raw != sw_lastStable){
    sw_lastStable = raw;
    if (!sw_lastStable){
      sw_pressMs = now; sw_longReported = false;
    } else {
      if (!sw_longReported){
        sendProgramChange(1, currentPrg);
        activity();
        lcdLineClear(1); lcdLinePrint16(1, "Sent PC (SHORT)");
      }
    }
  }
  if (!sw_lastStable && !sw_longReported && (now - sw_pressMs) >= LONG_MS){
    sw_longReported = true;
    currentPrg = 0; pendingSend = false;
    sendProgramChange(1, currentPrg);
    activity();
    lcdShowProgram();
    lcdLineClear(1); lcdLinePrint16(1, "Reset->PC#0 LONG");
  }
}

void setup(){
  pinMode(ACT_LED, OUTPUT); digitalWrite(ACT_LED, LOW);

  Wire.begin();
  lcd.init(); lcd.noAutoscroll(); lcd.noCursor(); lcd.noBlink(); lcd.backlight(); lcd.clear();
  lcdLinePrint16(0, "PRG:  0         ");
  lcdLinePrint16(1, "BTN: Ready      ");
  delay(30); lcdLineClear(0); lcdLineClear(1);
  lcdLinePrint16(0, "PRG:  0         ");
  lcdLinePrint16(1, "BTN: Ready      ");

  pinMode(ENC_CLK, INPUT_PULLUP);
  pinMode(ENC_DT,  INPUT_PULLUP);
  pinMode(ENC_SW,  INPUT_PULLUP);
  attachInterrupt(digitalPinToInterrupt(ENC_CLK), onEncCLKRise, RISING);

  Serial.begin(MIDI_BAUD);
  delay(300);
}

void loop(){
  unsigned long now = millis();

  long steps; noInterrupts(); steps = encSteps; encSteps = 0; interrupts();
  if (steps != 0){
    long v = (long)currentPrg + steps;
    if (v < 0) v = 0; else if (v > 127) v = 127;
    if ((uint8_t)v != currentPrg){
      currentPrg = (uint8_t)v;
      lcdShowProgram();
    }
    pendingSend = true;
    lastMoveMs = now;
  }

  if (pendingSend && (now - lastMoveMs) >= COMMIT_IDLE_MS){
    pendingSend = false;
    sendProgramChange(1, currentPrg);
    activity();
    lcdLineClear(1); lcdLinePrint16(1, "Sent PC (IDLE)");
  }

  scanButton();
  delay(1);
}
