\
// ---------------------------------------------------------
// Nano Ardule MIDI Controller
// Step 02: Rotary Encoder & Buttons Input Test
// 작성일: 2025-08-21
// ---------------------------------------------------------

#include <Wire.h>
#include <LiquidCrystal_I2C.h>

// ===== LCD =====
#define LCD_ADDR 0x27
#define LCD_COLS 16
#define LCD_ROWS 2
LiquidCrystal_I2C lcd(LCD_ADDR, LCD_COLS, LCD_ROWS);

// ===== Rotary Encoder (인터럽트 가능 핀) =====
const uint8_t ENC_CLK = 2;   // D2
const uint8_t ENC_DT  = 3;   // D3
const uint8_t ENC_SW  = 4;   // D4 (버튼)

volatile long encCount = 0;
volatile uint8_t lastCLK = 0;

// ===== Buttons (풀업 가정: GND로 눌림) =====
// 설계 요약 문서 기준
const uint8_t BTN_SPLIT = 5;   // D5
const uint8_t BTN_STOP  = 6;   // D6
const uint8_t BTN_SAVE  = 7;   // D7
const uint8_t BTN_LOAD  = 8;   // D8
const uint8_t BTN_PLAY  = A3;  // A3: 디지털로 사용 가능, 내부 풀업 OK → 외부 풀업 필요 없음
// ⚠ A6는 Nano에서 "아날로그 입력 전용"이라 디지털 풀업을 쓸 수 없습니다.
//    따라서 PART SELECT 버튼은 반드시 외부 풀업 저항(10k~47k) 필요!
const uint8_t BTN_PART_ANALOG = A6; // PART SELECT

// ===== 버튼 처리 파라미터 =====
const uint16_t DEBOUNCE_MS = 30;
const uint16_t LONGPRESS_MS = 700;

// 공통 상태 기록용
struct ButtonState {
  const char* name;
  uint8_t pin;
  bool isAnalog;        // A6 전용
  bool lastStable;      // 안정화된 상태 (true=HIGH, false=LOW)
  bool reading;         // 즉시 판독 결과
  unsigned long lastChangeMs;
  unsigned long pressStartMs;
  bool longReported;
};

// 버튼 목록(Encoder SW 포함)
ButtonState buttons[] = {
  {"ENC_SW",  ENC_SW,  false, true, true, 0, 0, false},
  {"SPLIT",   BTN_SPLIT, false, true, true, 0, 0, false},
  {"STOP",    BTN_STOP,  false, true, true, 0, 0, false},
  {"SAVE",    BTN_SAVE,  false, true, true, 0, 0, false},
  {"LOAD",    BTN_LOAD,  false, true, true, 0, 0, false},
  {"PLAY",    BTN_PLAY,  false, true, true, 0, 0, false},
  {"PART",    BTN_PART_ANALOG, true,  true, true, 0, 0, false},
};

// 최근 이벤트 표시
String lastEvent = "Ready";

// ===== 유틸: LCD 안전 출력(깜빡임 최소화) =====
void lcdPrintFixed(uint8_t col, uint8_t row, const String &s, uint8_t width) {
  lcd.setCursor(col, row);
  String out = s;
  if (out.length() < width) {
    out += String(' ', width - out.length());
  } else if (out.length() > width) {
    out.remove(width);
  }
  lcd.print(out);
}

// ===== Encoder ISR =====
void IRAM_ATTR onEncCLKChange() {
  // CLK 변화를 기준으로 DT 읽어서 방향 판단
  uint8_t clk = digitalRead(ENC_CLK);
  if (clk == lastCLK) return;
  lastCLK = clk;
  uint8_t dt = digitalRead(ENC_DT);
  if (clk == HIGH) {
    if (dt == LOW) encCount++;   // 시계방향
    else           encCount--;   // 반시계
  }
}

void setup() {
  // LCD
  lcd.init();
  lcd.backlight();
  lcd.clear();
  lcdPrintFixed(0, 0, "Step02 Inputs", 16);
  lcdPrintFixed(0, 1, "Init...", 16);
  delay(500);

  // Encoder
  pinMode(ENC_CLK, INPUT_PULLUP);
  pinMode(ENC_DT,  INPUT_PULLUP);
  lastCLK = digitalRead(ENC_CLK);
  attachInterrupt(digitalPinToInterrupt(ENC_CLK), onEncCLKChange, CHANGE);
  pinMode(ENC_SW, INPUT_PULLUP); // 버튼: 풀업

  // Buttons
  pinMode(BTN_SPLIT, INPUT_PULLUP);
  pinMode(BTN_STOP,  INPUT_PULLUP);
  pinMode(BTN_SAVE,  INPUT_PULLUP);
  pinMode(BTN_LOAD,  INPUT_PULLUP);
  pinMode(BTN_PLAY,  INPUT_PULLUP);
  // A6는 내부 풀업 불가 → 외부 풀업 사용 권장. 여기서는 아날로그 판독 사용.

  lcd.clear();
  lcdPrintFixed(0, 0, "ENC: 0", 16);
  lcdPrintFixed(0, 1, "Evt: Ready", 16);
}

// 아날로그 버튼(A6) 판독 → true=HIGH(유휴), false=LOW(눌림)로 표준화
bool readAnalogAsDigital(uint8_t analogPin) {
  int v = analogRead(analogPin);  // 0~1023
  // 외부 풀업 기준: 유휴시 HIGH(≈1023), 눌림시 GND(≈0)
  // 임계값 512 사용
  return (v > 512);  // true=HIGH, false=LOW
}

void handleButton(ButtonState &b, unsigned long now) {
  // 즉시 판독
  if (b.isAnalog) {
    b.reading = readAnalogAsDigital(b.pin);
  } else {
    b.reading = digitalRead(b.pin); // INPUT_PULLUP → 유휴 HIGH, 눌림 LOW
  }

  // 디바운스: 변화 감지
  static bool lastRaw[sizeof(buttons)/sizeof(buttons[0])] = {true};
  static unsigned long lastRawChangeMs[sizeof(buttons)/sizeof(buttons[0])] = {0};
  uint8_t idx = &b - buttons;

  if (b.reading != lastRaw[idx]) {
    lastRaw[idx] = b.reading;
    lastRawChangeMs[idx] = now;
  }

  // 안정화
  if ((now - lastRawChangeMs[idx]) >= DEBOUNCE_MS) {
    if (b.lastStable != b.reading) {
      // 상태가 안정적으로 바뀜
      b.lastStable = b.reading;
      b.lastChangeMs = now;

      if (b.lastStable == false) {
        // 눌림 시작(LOW)
        b.pressStartMs = now;
        b.longReported = false;
      } else {
        // 해제(HIGH): 길이 판단
        unsigned long pressDur = now - b.pressStartMs;
        if (!b.longReported) {
          if (pressDur >= LONGPRESS_MS) {
            lastEvent = String(b.name) + " LONG";
          } else {
            lastEvent = String(b.name) + " SHORT";
          }
        }
      }
    }

    // 길게 누르는 동안 한 번만 LONG 이벤트 보고
    if (b.lastStable == false && !b.longReported) {
      unsigned long held = now - b.pressStartMs;
      if (held >= LONGPRESS_MS) {
        lastEvent = String(b.name) + " LONG";
        b.longReported = true;
      }
    }
  }
}

void loop() {
  static long lastShownCount = 0;
  static unsigned long lastLCDms = 0;
  unsigned long now = millis();

  // 모든 버튼 스캔
  for (size_t i = 0; i < sizeof(buttons)/sizeof(buttons[0]); ++i) {
    handleButton(buttons[i], now);
  }

  // 인코더 카운트 표시(변경 시만 LCD 갱신)
  long c;
  noInterrupts();
  c = encCount;
  interrupts();
  if (c != lastShownCount || (now - lastLCDms) > 250) {
    lastShownCount = c;
    lastLCDms = now;
    char buf[17];
    snprintf(buf, sizeof(buf), "ENC: %ld", c);
    lcdPrintFixed(0, 0, String(buf), 16);
    lcdPrintFixed(0, 1, "Evt: " + lastEvent, 16);
  }

  // 루프는 빠르게 돌되, I2C 트래픽 과다 방지
  delay(1);
}
