\
// ---------------------------------------------------------
// Nano Ardule - Step 02 Final Inputs (Low Flicker + Buttons)
// - Encoder: D2 (CLK, interrupt), D3 (DT), RISING + guard
// - Buttons: D4 ENC_SW, D5 SPLIT, D6 STOP, D7 SAVE, D8 LOAD, A3 PLAY, A6 PART(analog, 10k pullup)
// - LCD: 1602 I2C (0x27), diff-update to minimize flicker
// 작성: 2025-08-24
// 튜닝 포인트:
// - LCD 업데이트 간격: LCD_MIN_MS를 150 → 200~250ms로 늘리면 더 부드럽게 보여요.
// - 엔코더 가드: STEP_GUARD_US를 600–1000µs에서 환경에 맞게.
// - A6 임계값: A6_THRESH_HIGH 850 / LOW 200 → 필요하면 800/250 등으로 미조정.
// ---------------------------------------------------------

#include <Wire.h>
#include <LiquidCrystal_I2C.h>

// ===== LCD =====
#define LCD_ADDR 0x27
#define LCD_COLS 16
#define LCD_ROWS 2
LiquidCrystal_I2C lcd(LCD_ADDR, LCD_COLS, LCD_ROWS);

// ===== Encoder Pins =====
const uint8_t ENC_CLK = 2;   // D2, interrupt
const uint8_t ENC_DT  = 3;   // D3
const uint8_t ENC_SW  = 4;   // D4 (encoder push)

// ===== Other Buttons =====
const uint8_t BTN_SPLIT = 5;   // D5
const uint8_t BTN_STOP  = 6;   // D6
const uint8_t BTN_SAVE  = 7;   // D7
const uint8_t BTN_LOAD  = 8;   // D8
const uint8_t BTN_PLAY  = A3;  // A3 (digital OK)
const uint8_t BTN_PART_ANALOG = A6; // A6 (analog-only, 10k pull-up to 5V)

// ===== Timing/Filter =====
const uint16_t DEBOUNCE_MS    = 30;
const uint16_t LONGPRESS_MS   = 700;
const uint16_t STEP_GUARD_US  = 800; // encoder ISR guard
const uint16_t LCD_MIN_MS     = 150; // min interval between ENC updates

// ===== A6 hysteresis thresholds =====
const int A6_THRESH_HIGH = 850; // rising -> HIGH (idle)
const int A6_THRESH_LOW  = 200; // falling -> LOW (pressed)

// ===== Button State Struct (declare BEFORE functions) =====
struct ButtonState {
  const char* name;
  uint8_t pin;
  bool isAnalog;
  bool lastStable;           // HIGH idle, LOW pressed
  bool reading;
  unsigned long lastChangeMs;
  unsigned long pressStartMs;
  bool longReported;
};

// Forward declarations
void lcd_write_line_diff(uint8_t row, const char* text);
void lcd_force_line(uint8_t row, const char* text);
bool readAnalogAsDigital(uint8_t analogPin, bool &hysteresisState);
void handleButton(ButtonState &b, unsigned long nowMs, char* eventBuf, size_t eventBufLen, bool &eventOccurred);
void onEncCLKRise();

// ===== LCD shadow buffers for diff update =====
char lcdShadow[2][LCD_COLS + 1] = { "                ", "                " };

void lcd_write_line_diff(uint8_t row, const char* text) {
  // Prepare 16-char padded buffer
  char buf[LCD_COLS + 1];
  snprintf(buf, sizeof(buf), "%-16s", text);

  // Compare and only write differing chars
  for (uint8_t i = 0; i < LCD_COLS; ++i) {
    if (lcdShadow[row][i] != buf[i]) {
      lcd.setCursor(i, row);
      lcd.print(buf[i]);
      lcdShadow[row][i] = buf[i];
    }
  }
}

void lcd_force_line(uint8_t row, const char* text) {
  char buf[LCD_COLS + 1];
  snprintf(buf, sizeof(buf), "%-16s", text);
  lcd.setCursor(0, row);
  lcd.print(buf);
  // update shadow entirely
  for (uint8_t i = 0; i < LCD_COLS; ++i) lcdShadow[row][i] = buf[i];
}

// ===== Encoder ISR =====
volatile long encCount = 0;
volatile unsigned long lastStepUs = 0;

void onEncCLKRise() {
  unsigned long now = micros();
  if ((unsigned long)(now - lastStepUs) < STEP_GUARD_US) return;
  lastStepUs = now;
  // Direction by DT on CLK rising
  if (digitalRead(ENC_DT) == LOW) encCount++;
  else                            encCount--;
}

// ===== Analog read w/ hysteresis for A6 =====
bool readAnalogAsDigital(uint8_t analogPin, bool &hysteresisState) {
  int v = analogRead(analogPin);
  if (hysteresisState) {
    // currently HIGH, only go LOW if well below LOW threshold
    if (v < A6_THRESH_LOW) hysteresisState = false;
  } else {
    // currently LOW, only go HIGH if well above HIGH threshold
    if (v > A6_THRESH_HIGH) hysteresisState = true;
  }
  return hysteresisState; // true=HIGH idle, false=LOW pressed
}

// ===== Buttons =====
ButtonState buttons[] = {
  {"ENC_SW",  ENC_SW, false, true, true, 0, 0, false},
  {"SPLIT",   BTN_SPLIT, false, true, true, 0, 0, false},
  {"STOP",    BTN_STOP,  false, true, true, 0, 0, false},
  {"SAVE",    BTN_SAVE,  false, true, true, 0, 0, false},
  {"LOAD",    BTN_LOAD,  false, true, true, 0, 0, false},
  {"PLAY",    BTN_PLAY,  false, true, true, 0, 0, false},
  {"PART",    BTN_PART_ANALOG, true,  true, true, 0, 0, false},
};

bool a6HysteresisState = true; // start HIGH (idle)

void handleButton(ButtonState &b, unsigned long nowMs, char* eventBuf, size_t eventBufLen, bool &eventOccurred) {
  // Raw reading
  bool rawHigh;
  if (b.isAnalog) {
    rawHigh = readAnalogAsDigital(b.pin, a6HysteresisState);
  } else {
    rawHigh = digitalRead(b.pin); // INPUT_PULLUP
  }

  // Debounce
  static bool lastRaw[sizeof(buttons)/sizeof(buttons[0])] = {true};
  static unsigned long lastRawMs[sizeof(buttons)/sizeof(buttons[0])] = {0};
  uint8_t idx = &b - buttons;

  if (rawHigh != lastRaw[idx]) {
    lastRaw[idx] = rawHigh;
    lastRawMs[idx] = nowMs;
  }
  if ((nowMs - lastRawMs[idx]) < DEBOUNCE_MS) return;

  // Stable transition
  if (b.lastStable != rawHigh) {
    b.lastStable = rawHigh;
    if (!b.lastStable) {
      // pressed (LOW)
      b.pressStartMs = nowMs;
      b.longReported = false;
    } else {
      // released (HIGH)
      unsigned long dur = nowMs - b.pressStartMs;
      if (!b.longReported) {
        snprintf(eventBuf, eventBufLen, "%s %s", b.name, (dur >= LONGPRESS_MS ? "LONG" : "SHORT"));
        eventOccurred = true;
      }
    }
  }

  // long-press during hold (report once)
  if (!b.lastStable && !b.longReported) {
    unsigned long held = nowMs - b.pressStartMs;
    if (held >= LONGPRESS_MS) {
      b.longReported = true;
      snprintf(eventBuf, eventBufLen, "%s LONG", b.name);
      eventOccurred = true;
    }
  }
}

// ===== Setup & Loop =====
void setup() {
  Wire.begin();
  // Wire.setClock(100000); // default 100kHz
  lcd.init();
  lcd.noAutoscroll();
  lcd.noCursor();
  lcd.noBlink();
  lcd.backlight();
  lcd.clear();

  // init shadow to spaces
  for (uint8_t r=0; r<2; ++r) for (uint8_t i=0;i<LCD_COLS;++i) lcdShadow[r][i] = ' ';

  // Encoder
  pinMode(ENC_CLK, INPUT_PULLUP);
  pinMode(ENC_DT,  INPUT_PULLUP);
  attachInterrupt(digitalPinToInterrupt(ENC_CLK), onEncCLKRise, RISING);

  // Buttons
  pinMode(ENC_SW, INPUT_PULLUP);
  pinMode(BTN_SPLIT, INPUT_PULLUP);
  pinMode(BTN_STOP,  INPUT_PULLUP);
  pinMode(BTN_SAVE,  INPUT_PULLUP);
  pinMode(BTN_LOAD,  INPUT_PULLUP);
  pinMode(BTN_PLAY,  INPUT_PULLUP);
  // A6 analog uses external 10k pull-up; no pinMode needed

  // Initial lines (force write twice to sanitize odd backpacks)
  lcd_force_line(0, "ENC:+0000000   ");
  lcd_force_line(1, "BTN: Ready     ");
  delay(30);
  lcd_force_line(0, "ENC:+0000000   ");
  lcd_force_line(1, "BTN: Ready     ");
}

void loop() {
  static long shown = 0;
  static unsigned long lastLCDms = 0;
  static unsigned long eventShownUntil = 0;
  static bool showingEvent = false;

  unsigned long nowMs = millis();

  // --- Scan buttons ---
  char eventBuf[17] = {0};
  bool eventOccurred = false;
  for (size_t i=0; i<sizeof(buttons)/sizeof(buttons[0]); ++i) {
    handleButton(buttons[i], nowMs, eventBuf, sizeof(eventBuf), eventOccurred);
  }
  if (eventOccurred) {
    char line[17];
    snprintf(line, sizeof(line), "BTN: %-12s", eventBuf);
    lcd_write_line_diff(1, line);
    showingEvent = true;
    eventShownUntil = nowMs + 1000; // show for 1s
  }
  // One-shot revert
  if (showingEvent && (long)(nowMs - eventShownUntil) >= 0) {
    lcd_write_line_diff(1, "BTN: Ready     ");
    showingEvent = false;
  }

  // --- Encoder display ---
  long v; noInterrupts(); v = encCount; interrupts();
  if ( (v != shown && (nowMs - lastLCDms) >= LCD_MIN_MS) || (nowMs - lastLCDms) >= 500 ) {
    shown = v;
    lastLCDms = nowMs;
    char line[17];
    // fixed width, signed with zero padding
    // ENC:+000123
    snprintf(line, sizeof(line), "ENC:%+07ld     ", v);
    lcd_write_line_diff(0, line);
  }

  delay(1);
}
