#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
adpgen.py (APT v2.1 strict)
- ADT 텍스트(그리드)를 ADP(바이너리)로 변환
- 허용 기호: '^', 'X/x'(=2), 'O/o'(=1), '.'(=0)  (레거시 미해석, 등장 시 오류)
- 그리드 공백 및 '||'는 파싱 단계에서 제거
- SLOTS 순서대로 직렬화 (표시 순서와 무관)
- 헤더: steps(uint16), bpm(uint8), midi_ch(uint8)  뒤에 slots*steps 바이트의 ACC 배열
"""
import argparse, struct, re
from pathlib import Path
from typing import Dict, List, Tuple

ALLOWED = set(['^','X','x','O','o','.'])
ACC_MAP = {'^':3, 'X':2, 'x':2, 'O':1, 'o':1, '.':0}

def parse_slots(line: str):
    _, rhs = line.split('=', 1)
    parts = [p.strip() for p in rhs.split(',') if p.strip()]
    out = []
    for p in parts:
        name, nn = p.split(':', 1)
        out.append((name.strip(), int(nn.strip())))
    return out

def parse_adt(path: Path):
    with open(path, encoding='utf-8') as f:
        lines = [ln.rstrip('\n') for ln in f]

    steps = 32; bpm = 120; midi_ch = 9
    slots = []
    grid_by_name: Dict[str, str] = {}

    for ln in lines:
        if ln.startswith('STEPS='):
            m = re.search(r'STEPS=(\d+)', ln); 
            if m: steps = int(m.group(1))
            m = re.search(r'BPM=(\d+)', ln); 
            if m: bpm = int(m.group(1))
            m = re.search(r'MIDI_CH=(\d+)', ln); 
            if m: midi_ch = int(m.group(1))
        elif ln.startswith('SLOTS='):
            slots = parse_slots(ln)
        elif ':' in ln and not ln.startswith('SLOTS='):
            name, rest = ln.split(':', 1)
            grid_by_name[name.strip()] = rest

    def cleanse(s: str) -> str:
        return s.replace(' ', '').replace('|', '')

    rows = []
    for name, _nn in slots:
        raw = grid_by_name.get(name, '')
        flat = list(cleanse(raw))
        for i, ch in enumerate(flat):
            if ch not in ALLOWED:
                raise ValueError(f"Unsupported symbol '{ch}' in row '{name}' at pos {i}. Allowed: '^', 'X/x', 'O/o', '.'")
        if len(flat) < steps:
            flat += ['.'] * (steps - len(flat))
        else:
            flat = flat[:steps]
        rows.append([ACC_MAP[ch] for ch in flat])

    return steps, bpm, midi_ch, rows

def write_adp(infile: Path, outfile: Path):
    steps, bpm, midi_ch, rows = parse_adt(infile)
    with open(outfile, 'wb') as f:
        f.write(struct.pack('<HBB', steps, bpm, midi_ch))
        for r in rows:
            f.write(bytes(r))
    print(f"Wrote {outfile.name}  (steps={steps}, bpm={bpm}, ch={midi_ch}, slots={len(rows)})")

def main():
    ap = argparse.ArgumentParser(description="Convert ADT (APT v2.1 strict) grid to ADP binary.")
    ap.add_argument('infile', help='Input .ADT path')
    ap.add_argument('--out', default=None, help='Output .ADP path (default: same name)')
    args = ap.parse_args()
    infile = Path(args.infile)
    outfile = Path(args.out) if args.out else infile.with_suffix('.ADP')
    write_adp(infile, outfile)

if __name__ == '__main__':
    main()
